﻿# PyU3D Collision Example.
# Written by MysteriXYZ.

# Declare constants
SCREEN_WIDTH = 800
SCREEN_HEIGHT = 600
SPEED_LIMIT = 100

GRAVITY = 600. # downward acceleration in units per second squared

# Import the PyU3D package
import PyU3D as u3d
from PyU3D._local import *
import PyU3D.host as host

# a premade sphere model will be imported to visualize a swept ellipsoid
import PyU3D.custom as custom

# Psyco is strongly recommended to boost the framerate
import psyco
psyco.full()



# create a custom Camera class, derived from the PyU3D.Camera class
class Camera(u3d.Camera):

  """
  Camera with custom behaviour.

  This camera is aligned to ("follows") a pivot whose pitch is controlled by the
  mouse. This pivot is in turn aligned to a target object whose yaw is also
  controlled by the mouse.
  The alignment procedure provides a way to smooth out the camera motion,
  relative to that of its pivot (and that of the pivot, relative to that of the
  camera target).

  """

  def __init__(self, *args, **kwargs):

    # inherit from PyU3D.Camera
    super(Camera, self).__init__(*args, **kwargs)

    # initialize the camera target
    self.target = None

    # create the camera pivot
    self.pivot = u3d.Transformable()

    # define a yaw offset for alignment of the pivot to the target
    self.pivot_to_target = RotationMatrix(0., 69.5, 0.)

    # define a yaw offset for alignment of the pivot to the target
    self.pivot_translation = TranslationMatrix(0., -45., 0.)

    # define a yaw offset for alignment of the camera to the pivot
    self.cam_to_pivot = RotationMatrix(0., 20.5, 0.)

    # initialize the mouse speed to rotate the camera pivot and target
    self.mouse_speed = .1

    # create a ray to keep the camera out of obstacles
    self.ray = Ray()


  def attachTo(self, target):

    """Attach the camera to a target."""

    self.target = target

    # align the pivot to the target
    self.pivot.alignTo(
                        self.target, self.pivot_to_target,
                        # since the pitch of the pivot is controlled by the
                        # mouse and therefore independent from the pitch of the
                        # target, it must not be affected by the alignment
                        orientation=(False, True, False)
                      )

    # Currently, the pivot is aligned to the target using only a yaw offset,
    # but it also needs a position offset; if this were included in the offset
    # matrix for the alignTo(...) method, the result would look quite bad if
    # more smoothing were used for rotation than for translation (because the
    # position would not sufficiently correspond to the orientation then).
    # So to keep the pivot at a certain distance (in a certain direction) from
    # the target, an additional transformation of the pivot has to be
    # performed manually.

    # save the current pitch of the pivot
    pivot_pitch = self.pivot.pitch

    # temporarily change the pitch of the pivot...
    self.pivot.pitch = 19.5
    # ...so it can be offset in the desired direction by applying a
    # translation in local space
    self.pivot.transform(self.pivot_translation, True)

    # reset the pitch of the pivot to its previous value
    self.pivot.pitch = pivot_pitch

    # align the camera to its pivot
    self.alignTo(
                  self.pivot, self.cam_to_pivot,
                  orientation=(True, True, False)
                )

    self.update()


  def step(self):

    """Align camera to pivot and update its transformation."""

    mouse_x, mouse_y = mouse.getPosition()

    self.pivot.pitch += (mouse_y - SCREEN_HEIGHT / 2) * self.mouse_speed

    if self.pivot.pitch < -40.:
      self.pivot.pitch = -40.
    if self.pivot.pitch > 40.:
      self.pivot.pitch = 40.

    if self.target:

      self.target.yaw += (mouse_x - SCREEN_WIDTH / 2) * self.mouse_speed
      self.target.yaw %= 360.

    mouse.setPosition(SCREEN_WIDTH / 2, SCREEN_HEIGHT / 2)

    if self.target:

      self.target.step()

      smoothing_factor = min(1., 18. * host.getDeltaTime())

      # align the pivot to the target
      self.pivot.alignTo(
                          self.target, self.pivot_to_target,
                          # since the pitch of the pivot is controlled by the
                          # mouse and therefore independent from the pitch of
                          # the target, it must not be affected by the
                          # alignment
                          orientation=(False, True, False),
                          smoothness=(0., smoothing_factor, 0.)
                        )

      # Currently, the pivot is aligned to the target using only a yaw offset,
      # but it also needs a position offset; if this were included in the offset
      # matrix for the alignTo(...) method, the result would look quite bad if
      # more smoothing were used for rotation than for translation (because the
      # position would not sufficiently correspond to the orientation then).
      # So to keep the pivot at a certain distance (in a certain direction) from
      # the target, an additional transformation of the pivot has to be
      # performed manually.

      # save the current pitch of the pivot
      pivot_pitch = self.pivot.pitch

      # temporarily change the pitch of the pivot...
      self.pivot.pitch = 19.5
      # ...so it can be offset in the desired direction by applying a
      # translation in local space
      self.pivot.transform(self.pivot_translation, True)

      # reset the pitch of the pivot to its previous value
      self.pivot.pitch = pivot_pitch

      # align the camera to its pivot
      self.alignTo(
                    self.pivot, self.cam_to_pivot,
                    orientation=(True, True, False),
                    smoothness=(
                                smoothing_factor**1.2,
                                smoothing_factor,
                                0.
                               )
                  )

      # Keep the camera out of obstacles

      # compute the vector pointing from the target to the camera...
      x, y, z = self.target.getPosition()
      z += self.target.height * .5
      target_pos_vect = Vector(x, y, z)
      cam_offset_vect = Vector(self.getPosition()) - target_pos_vect

      # ... and use it to trace a ray from the target...
      self.ray.setPosition(x, y, z)
      # ... to the level in the direction of the camera
      self.ray.setDirection(cam_offset_vect.getDirection())

      if self.ray.trace():

        length = cam_offset_vect.getLength()
        dist = self.ray.getIntersectDist() - 3.

        # if the distance from the target to the level (minus a minimal offset)
        # is shorter than the distance to the camera, then the camera needs to
        # be moved closer to its target
        if dist < length:
          self.setPosition(target_pos_vect + cam_offset_vect.normalize() * dist)

    self.update()



class WereWolf(u3d.Model):

  def __init__(self, x=0., y=0., z=0.):

    # inherit from PyU3D.Model
    super(WereWolf, self).__init__(
                                    "mdl/wolf.md2", "",
                                    x, y, z,
                                    scalx=.4,
                                    texture=Texture("gfx/wolfskin.bmp")
                                  )

    self.speed = 100.
    self.falling_speed = 0.
    self.min_level_dist = 7. # prevents the werewolf from getting into walls
    self.height = 19.

    # the vertical offset of the origin of the werewolf model from its feet
    self.origin_z = 8.75

    # the vertical offset of the ellipsoid from the origin of the werewolf model
    offs_z = 3.

    # create a swept ellipsoid to manage collisions for the werewolf
    self.ellipsoid = SweptEllipsoid(
                                    self.x, self.y, self.z+offs_z,
                                    self.pitch, self.yaw, self.roll,
                                    self.min_level_dist, self.min_level_dist,
                                    self.height*.5
                                   )

    # the vertical offset of the ellipsoid from the feet of the werewolf model
    self.ellipsoid.offs_z = self.origin_z + offs_z

    # define an offset for the alignment of the werewolf to the ellipsoid
    # (after collision management);
    # make sure to divide it by the scaling of the ellipsoid, since aligning it
    # will multiply it by that scaling (the alignment happens in the local space
    # of the ellipsoid
    self.offs_matr = TranslationMatrix(0., 0., -offs_z/self.ellipsoid.scalz)

    # create a transparent sphere to visualize the swept ellipsoid
    self.ellipsoid.shape = custom.SphereModel()
    self.ellipsoid.shape.getMaterial().setDiffuseColor(255, 255, 255, 128)

    # create a ray that will be cast downwards, to check collisions with the
    # ground and to find out if jumping is possible
    self.downwards_ray = Ray(latitude=90.)


  def step(self):

    # since the werewolf will be aligned to the ellipsoid, preserve the yaw of
    # werewolf model
    self.ellipsoid.yaw = self.yaw

    x_prev, y_prev = self.x, self.y

    z_prev = self.ellipsoid.z

    delta_time = host.getDeltaTime()

    grav_per_sec = GRAVITY * delta_time

    self.falling_speed -= grav_per_sec

    # Determine the forward and sideways movement directions, based on the
    # keys that are held down

    speed = self.speed * delta_time

    advance = (keyb.keyIsDown("up") - keyb.keyIsDown("down")) * speed
    strafe = (keyb.keyIsDown("right") - keyb.keyIsDown("left")) * speed

    # Create the vector that describes the movement of the werewolf

    advance_vect = DirectionVector(0., self.yaw+90.) * advance
    strafe_vect = DirectionVector(0., self.yaw+180.) * strafe
    falling_vect = Vector(0., 0., self.falling_speed * delta_time)

    move_vect = advance_vect + strafe_vect + falling_vect

    # provisional animation control
    if (advance_vect + strafe_vect).getLength() > .1:
      self.updateAnimation(10. * delta_time, (40., 45.))
    else:
      self.updateAnimation(10. * delta_time, (0., 39.))

    # use the movement vector to move the swept ellipsoid;
    # the automatic collision management will set the ellipsoid to a position
    # that is collision-free, after attempting to move it into the direction
    # described by that vector, by an amount equal to the length of that vector
    self.ellipsoid.move(move_vect)

    self.downwards_ray.setPosition(self.ellipsoid.getPosition())

    if not self.downwards_ray.trace():

      vertical_offset = 100000.

      # prevent the werewolf from falling into the void
      self.falling_speed = 0.
      self.ellipsoid.z = z_prev

    else:

      # calculate the vertical offset of the feet of the werewolf from the
      # ground
      vertical_offset = self.downwards_ray.getIntersectDist() \
                        - self.ellipsoid.offs_z

      if vertical_offset < 0.:

        # prevent the werewolf from sinking into the ground
        self.ellipsoid.z -= vertical_offset
        self.falling_speed = 0.

      # sometimes it can happen that the vertical offset is bigger than the
      # distance from the feet down to the ground, yet the werewolf cannot get
      # any deeper; this can occur while moving along stairs for example, when
      # the ray is cast in-between two stair steps and detects the ground far
      # below instead of either step; in such a case, the falling speed would
      # constantly increase and movement would no longer be possible;
      # the following check can prevent this from happening
      elif (abs(self.ellipsoid.z - z_prev) < grav_per_sec * delta_time) \
        and (self.falling_speed < -grav_per_sec):

          self.falling_speed = -grav_per_sec
          vertical_offset = 0.

    # implement jumping
    if (mouse.buttonIsDown("right") or keyb.keyIsDown(" ")) \
      and vertical_offset <= 0.:
        self.falling_speed += 160. # jump speed in units per second

    # now the werewolf model can be aligned to the ellipsoid
    self.alignTo(self.ellipsoid, self.offs_matr)

    # update the transformation of the werewolf model
    self.update()

    # update the transformation of the sphere that visualizes the ellipsoid
    self.ellipsoid.shape.setTransformation(self.ellipsoid.getTransformation())
    self.ellipsoid.shape.update()



# Set U3D options

u3d.setLog("log.txt")
u3d.setZBufferFormat(32)
u3d.setMultiSampleType(2)


# initialize U3D and its host
host.init(
          SCREEN_WIDTH, SCREEN_HEIGHT, SPEED_LIMIT,
          "PyU3D Collision Example", "U3DIcon.png",
          fullscreen=False
         )


# create an object to handle keyboard input
keyb = host.Keyboard()

# create an object to handle the mouse
mouse = host.Mouse()

# hide the cursor
mouse.setVisibility(False)

# should problems arise due to the cursor being hidden, try to use a transparent
# cursor instead of hiding it; comment out the line above and de-comment the
# line below
##mouse.setCursor(None)


# Create the scene objects

# create the camera
camera = Camera(SCREEN_WIDTH, SCREEN_HEIGHT)

# create the directional light
dir_light = u3d.DirectionalLight()

# create a WereWolf object
werewolf = WereWolf(400., 336., 8.75)

# attach the camera to the werewolf
camera.attachTo(werewolf)

# define the U3D room to hide the swept ellipsoid in
invisible_room = u3d.Room()

#create the terrain
terrain = u3d.Terrain("gfx/TerrainHeightMap.png", 115, 10, 10, 64)
terrain.addTexture("gfx/TerrainStone.png",33,30)
terrain.addTexture("gfx/TerrainGras.png",40,25,"gfx/TerrainGrasAlpha.png")
terrain.addTexture("gfx/TerrainMud.png",35,29,"gfx/TerrainMudAlpha.png")
terrain.addTexture("gfx/TerrainSand.png",39,32,"gfx/TerrainSandAlpha.png")
terrain.createAtPos(48., 48., -26.)
terrain.setSolidity()

#create the house
house = u3d.Model("mdl/Room.an8", "", 416., 336.)
house.setSolidity()


loadBackgroundTex("gfx/sea2.jpg")


arial12 = Font("Arial", 12, BOLDITALIC)



# main loop
def main():

  while True:

    # process the current step of the game
    host.step()

    if keyb.keyIsPressed("esc"):
      host.exit()
    elif keyb.keyIsPressed("del"):
      werewolf.ellipsoid.shape.setRoom(invisible_room)
    elif keyb.keyIsPressed("enter"):
      werewolf.ellipsoid.shape.setRoom(PrimaryRoom)

    terrain.calculateLightMap(150)

    camera.step()

    arial12.draw(10, 10, "fps: "+str(host.getFPS()))
    arial12.draw(10, 40, "triangles: "+str(getDrawnTriangleCount()))
    arial12.draw(10, 70, "draw calls: "+str(u3d.getDrawCallCount()))
    arial12.draw(10, SCREEN_HEIGHT-80, "Press <Delete> to hide the ellipsoid.")
    arial12.draw(10, SCREEN_HEIGHT-50, "Press <Enter> to show the ellipsoid.")


if __name__ == '__main__':
  main()
